(* ::Package:: *)

(********1*********2*********3*********4*********5*********6*********7*****
 * KY_upperbound.m = Computing Killing curvatures and
 * solving the curvature conditions for a given metric
 * Package developed by Kei Yamamoto and edited by Tsuyoshi Houri
 * Version Number: v1.2 (13rd August, 2014)
 **************************************************************************)


BeginPackage["KYupperbound`"]

(***** Tensor manipulation *****)
contract::usage = "Contract ith and jth indices of a tensor of rank r, or contract i1th index of a tensor of rank r1 with i2th index of another tensor of rank r2."
skew::usage = "Anti-symmerise the indices specified in the list tIndices of a tensor of rank r."
cycle::usage = "Take a cyclic average. Used when antisymmetrising a 1-form with a p-form."
skewDelta::usage = "Generate a generalised Kronecker's delta of rank 2p in dimension d."

(***** Differential calculus *****)
exteriorD::usage = "Take exterior derivative of a p-form with respect to coordinates x."
(* Arguments: a p-form in a coordinate basis, p (rank of the p-form), the coordinates, the number of non-skew indices *)
(* The input should be a (i,j+p) tensor where the last p indices are skew symmetric. *)

(***** Utility functions *****)
combinations::usage = "Generate all possible combinations for d choose p."
getZeros::usage = "Generate a dCp by dCp-1 zero matrix."

(***** Riemannian geometry *****)
connectionOneForm::usage = "Compute connection 1-form in a tetrad basis."
curvatureTwoForm::usage = "Compute curvature 2-form in a tetrad basis."
covariantD::usage = "Take covariant derivative of a tensor of rank r with uiNumber upper indices in a tetrad basis."
(* Arguments: a tensor in a tetrad basis, rank of the tensor, the number of upper indices, metric in the tetrad basis, vielbein in a coordinate basis, the coordinates *)
(* The input tensor indices should be sorted so that all the upper indices appear to the left *)
star::usage = "Hodge dual operation."
(* Arguments: a p-form in any basis, metric in the basis, dimension, p (rank of the p-form), the number of non-skew indices *)
(* The input should be a (i,j+p) tensor where the last p indices are skew symmetric. *)
christoffel::usage = "Compute Christoffel symbol in a coordinate basis."
riemann::usage = "Compute Riemann tensor in a coordinate basis."
ricci::usage = "Compute Ricci tensor for a Riemann tensor."
einstein::usage = "Compute Einstein tensor for a given metric and Riemann tensor."

(***** Kiling-Yano upperbound *****)
curvatureOnForms::usage = "Curvature operator acting on p-forms."
(* RXXp of the original version is given by curvatureOnForms[curvatureTwoForm[g,e,x],0,d,p] *)
(* NxRXXp of the original version is given by curvatureOnFomrs[covD[curvatureTwoForm[g,e,x],4,1,g,e,x],1,d,p] *)
rPX::usage = "Auxiliary function."
(* RPlusXp of the original version is given by rPX[curvatureOnForms[curvatureTwoForm[g,e,x],0,d,p],0,d,p] *)
(* NxRPlusXp of the original version is given by rPX[curvatureOnFomrs[covD[curvatureTwoForm[g,e,x],4,1,g,e,x],1,d,p],1,d,p] *)
rMX::usage = "Auxiliary function."
(* RMinusXp of the original version is given by rMX[curvatureOnForms[curvatureTwoForm[g,e,x],0,d,p],0,d,p,g] *)
(* NxRPlusXp of the original version is given by rMX[curvatureOnFomrs[covD[curvatureTwoForm[g,e,x],4,1,g,e,x],1,d,p],1,d,p,g] *)
qR::usage = "Auxiliary function."
(* qRp of the original version is given by qR[curvatureOnForms[curvatureTwoForm[g,e,x],0,d,p],0,d,p,g] *)
(* qNxRp of the original version is given by qR[curvatureOnFomrs[covD[curvatureTwoForm[g,e,x],4,1,g,e,x],1,d,p],1,d,p,g] *)

(***** Blocks of Killing curvature for CCKY tensors *****)
M11::usage = "p-form -> p-form"
M21::usage = "p-form -> p-1-form"
M22::usage = "p-1-form -> p-1-form"

(***** Blocks of Killing curvature for KY tensors *****)
N11::usage = "p-form -> p-form"
N21::usage = "p-form -> p+1-form"
N22::usage = "p+1-form -> p+1-form"

(***** Matrix (rank 2) representation of Killing curvatures *****)
kcCCKY::usage = "Get matrix representations of Killing curvature for CCKY tensors."
kcKY::isage = "Get matrix representations of Killing curvature for KY tensors."

(***** Solving kernel *****)
solveCCKY::usage = "Solve the kernel of the Killing curvature for CCKY tensors."
solveKY::usage = "Solve the kernel of the Killing curvature for KY tensors."

(***** help command *****)
helpKYupperbound::usage = "help command, which lists the functions."


Begin["Private`"]

(* Tensor manipulation *)

contract[t_List,r_Integer,i_Integer,j_Integer]:=
	Tr[Transpose[t,{Sequence@@Range[3,i+1],1,Sequence@@Range[i+2,j],2,Sequence@@Range[j+1,r]}],Plus,2];
contract[t1_List,t2_List,r1_Integer,r2_Integer,i1_Integer,i2_Integer]:=
	Dot[Transpose[t1,{Sequence@@Range[1,i1-1],r1,Sequence@@Range[i1,r1-1]}],Transpose[t2,{Sequence@@Range[2,i2],1,Sequence@@Range[i2+1,r2]}]];

skew[t_List,r_Integer,tIndices_List]:=
	Module[{perms,myList},
			perms=Rest[Permutations[tIndices]];
			myList=Transpose[{Signature/@perms,Thread[Rule[tIndices,#]]&/@perms}];
			Fold[Plus[#1,#2[[1]]*Transpose[t,Range[r]/.#2[[2]]]]&,t,myList]/(Length[tIndices]!)
	];

cycle[t_List,r_Integer,k_Integer,tIndices_List]:=
	Module[{indices,l,perms,myList},
			indices=Union[{k},tIndices];
			l=Length[indices];
			perms=Rest[FoldList[RotateRight[#1,#2]&,indices,Table[1,{l-1}]]];
			myList=Transpose[{Signature/@perms,Thread[Rule[indices,#]]&/@perms}];
			Fold[Plus[#1,#2[[1]]*Transpose[t,Range[r]/.#2[[2]]]]&,t,myList]/l
	];

skewDelta[d_Integer,p_Integer?Positive]:=
	Module[{deltas},
			deltas=Outer[Times,Sequence@@Table[IdentityMatrix[d],{p}]];
			skew[deltas,2p,2*Range[p]]
	];
skewDelta[d_Integer,0]:=1;

exteriorD[f_List,r_Integer,x_List,offset_Integer]:=
	Module[{temp},
			temp=Outer[D[#1,#2]&,f,x];
			(-1)^r*(r+1)*skew[temp,r+offset+1,Range[r+1]+offset]
	];

combinations[d_Integer,p_Integer]:=
	Union[Sort/@Permutations[Range[d],{p}]];

getZeros[d_Integer,p_Integer]:=
	Table[0,{i,d!/(p!*(d-p)!)},{j,d!/((p-1)!*(d-p+1)!)}];

(* Riemannian geometry *)

connectionOneForm[g_,e_,x_]:=
	Module[{ig,ie,temp1,temp2,temp3},
			ig=Inverse[g];
			ie=Inverse[e];
			temp1=contract[contract[exteriorD[e,1,x,1],ie,3,2,2,1],ie,3,2,2,1];
			temp2=contract[ig,contract[g,temp1,2,3,2,1],2,3,2,3];
			temp3=Transpose[temp2,{1,3,2}];
			Simplify[(temp1+temp2+temp3)/2]
	];

curvatureTwoForm[g_,e_,x_]:=
	Module[{con,temp},
			con=connectionOneForm[g,e,x];
			temp=contract[Outer[D[#1,#2]&,con,x],Inverse[e],4,2,4,1]-contract[con,con,3,3,3,1]-Transpose[contract[con,con,3,3,2,1],{1,3,2,4}];
			Simplify[-temp+Transpose[temp,{1,2,4,3}]]
	];

covariantD[t_List,r_Integer,uiNumber_Integer,g_List,e_List,x_List]:=
	Module[{con,temp1,temp2},
			con=connectionOneForm[g,e,x];
			temp1=contract[Outer[D[#1,#2]&,t,x],Inverse[e],r+1,2,r+1,1];
			temp2=Fold[Plus[#1,Transpose[contract[t,con,r,3,#2,2],{Sequence@@Delete[Range[r],#2],#2,r+1}]]&,temp1,Range[uiNumber]];
			Simplify[Fold[Plus[#1,-Transpose[contract[t,con,r,3,#2,1],{Sequence@@Delete[Range[r],#2],#2,r+1}]]&,temp2,Range[uiNumber+1,r]]]
	];

star[f_List,g_List,d_Integer,p_Integer,offset_Integer]:=
	Module[{ig,temp},
			ig=Inverse[g];
			temp=Fold[contract[#1,ig,p+offset,2,#2,1]&,f,Table[1+offset,{p}]];
			temp=Sqrt[-Det[g]]*contract[temp,Normal[LeviCivitaTensor[d]],p+offset,d,1+offset,1];
			1/p!*Fold[contract[#1,p+d+offset-2-2*#2,1+offset,p+offset-#2]&,temp,Range[p-1]-1]
	];

christoffel[g_List,x_List]:=
	Module[{temp},
		temp=Outer[D,g,x];
		Simplify[Inverse[g].(temp+Transpose[temp,{1,3,2}]-Transpose[temp,{3,2,1}])/2]
	];

riemann[g_List,x_List]:=
	Module[{chris,temp1,temp2},
		chris=christoffel[g,x];
		temp1=Outer[D,chris,x];
		temp2=chris.chris;
		Simplify[Transpose[temp1,{1,2,4,3}]-temp1+Transpose[temp2,{1,3,2,4}]-Transpose[temp2,{1,4,2,3}]]
	];

ricci[r_List]:=
	Simplify[contract[r,4,1,3]];

einstein[g_,r_List]:=
	Module[{ric},
			ric=ricci[r];
			Simplify[ric-1/2*Simplify[Tr[Dot[ric,Inverse[g]]]]*g]
	];

(* Killing-Yano upperbound *)

curvatureOnForms[c_List,n_Integer,d_Integer,p_Integer?Positive]/;d>p:=
	Module[{temp1,temp2,temp3},
			temp1=Outer[Times,Transpose[c,{3+n,4+n,1+n,2+n,Sequence@@Range[n]}],skewDelta[d,p-1]];
			temp2=cycle[temp1,n+2*p+2,3+n,Range[5+n,n+2*p+2,2]];
			temp3=cycle[temp2,n+2*p+2,4+n,Range[6+n,n+2*p+2,2]];
			Simplify[-p*(p!)*Transpose[temp3,{Sequence@@Range[1,n+2],Sequence@@Flatten[{#,#+p}&/@Range[3+n,n+p+2]]}]]
	];
curvatureOnForms[c_List,n_Integer,d_Integer,0]:=0;
curvatureOnForms[c_List,n_Integer,d_Integer,d_Integer]:=0;

rPX[c_List,n_Integer,d_Integer,p_Integer]:=
	Simplify[(p+1)*Transpose[cycle[c,n+2*p+2,2+n,Range[3+n+p,n+2*p+2]],{Sequence@@Range[1+n],n+p+2,Sequence@@Range[2+n,n+p+1],Sequence@@Range[n+p+3,n+2*p+2]}]];

rMX[c_List,n_Integer,d_Integer,p_Integer,g_List]:=
	Module[{temp},
			temp=contract[c,Inverse[g],n+2*p+2,2,2+n,1];
			Simplify[contract[temp,n+2*p+2,n+p+2,n+2*p+2]]
	];

qR[c_List,n_Integer,d_Integer,p_Integer,g_List]:=
	Module[{cofp,temp},
			cofp=rPX[c,n,d,p];
			temp=contract[Inverse[g],cofp,2,n+2*p+2,2,1+n];
			Simplify[-contract[temp,n+2*p+2,1,n+p+2]]];

M11[g_,e_,x_,p_]:=
	Module[{dim,curv,temp},
			dim=Length[x];
			curv=curvatureOnForms[curvatureTwoForm[g,e,x],0,dim,p];
			temp=2*skew[Transpose[Outer[Times,rMX[curv,0,dim,p,g],g],{Sequence@@Range[2,p+2],Sequence@@Range[p+4,2*p+2],1,p+3}],2*p+2,{1,2}];
			Simplify[curv+p/(dim-p)*cycle[temp,2*p+2,2*p+2,Range[p+3,2*p+1]]]
	];

M21[g_,e_,x_,p_]:=
	Module[{dim,curvD},
			dim=Length[x];
			curvD=covariantD[curvatureTwoForm[g,e,x],4,1,g,e,x];
			Simplify[(2*(dim-p+1))/(dim-p)*skew[rMX[curvatureOnForms[curvD,1,dim,p],1,dim,p,g],2*p+1,{1,2}]]
	];

M22[g_,e_,x_,p_]:=
	Module[{dim,curv},
			dim=Length[x];
			curv=curvatureOnForms[curvatureTwoForm[g,e,x],0,dim,p];
			Simplify[curvatureOnForms[curvatureTwoForm[g,e,x],0,dim,p-1]-2/(dim-p)*skew[contract[g,rMX[curv,0,dim,p,g],2,2*p,2,2],2*p,{1,2}]]
	];

N11[g_,e_,x_,p_]:=
	Module[{dim,curv},
			dim=Length[x];
			curv=curvatureOnForms[curvatureTwoForm[g,e,x],0,dim,p];
			Simplify[curv+2/p*skew[Transpose[rPX[curv,0,dim,p],{Sequence@@Range[2,p+2],1,Sequence@@Range[p+3,2*p+2]}],2*p+2,{1,2}]]
	];

N21[g_,e_,x_,p_]:=
	Module[{dim,curvD},
			dim=Length[x];
			curvD=covariantD[curvatureTwoForm[g,e,x],4,1,g,e,x];
			Simplify[-((2*(p+1))/p)*skew[rPX[curvatureOnForms[curvD,1,dim,p],1,dim,p],2*p+3,{1,2}]]
	];

N22[g_,e_,x_,p_]:=
	Module[{dim,curv},
			dim=Length[x];
			curv=Transpose[Outer[Times,IdentityMatrix[dim],rPX[curvatureOnForms[curvatureTwoForm[g,e,x],0,dim,p],0,dim,p]],{2,3,1,Sequence@@Range[4,2*p+4]}];
			Simplify[curvatureOnForms[curvatureTwoForm[g,e,x],0,dim,p+1]+(2*(p+1))/p*cycle[skew[curv,2*p+4,{1,2}],2*p+4,3,Range[4,p+3]]]
	];

kcCCKY[g_,e_,x_,i_Integer,j_Integer,p_Integer]:=
	Module[{dim,combs1,combs2,temp,m11,m21,m22},
			dim=Length[x];
			combs1=combinations[dim,p];
			combs2=combinations[dim,p-1];
			temp=M11[g,e,x,p];
			m11=Simplify[Outer[temp[[i,j,Sequence@@#2,Sequence@@#1]]&,combs1,combs1,1]];
			temp=M21[g,e,x,p];
			m21=Simplify[Outer[temp[[i,j,Sequence@@#2,Sequence@@#1]]&,combs2,combs1,1]];
			temp=M22[g,e,x,p];
			m22=Simplify[Outer[temp[[i,j,Sequence@@#2,Sequence@@#1]]&,combs2,combs2,1]];
			Transpose[Join[Transpose[Join[m11,m21]],Transpose[Join[getZeros[dim,p],m22]]]]
	];

kcKY[g_,e_,x_,i_Integer,j_Integer,p_Integer]:=
	Module[{dim,combs1,combs2,temp,n11,n21,n22},
			dim=Length[x];
			combs1=combinations[dim,p];
			combs2=combinations[dim,p+1];
			temp=N11[g,e,x,p];
			n11=Simplify[Outer[temp[[i,j,Sequence@@#2,Sequence@@#1]]&,combs1,combs1,1]];
			temp=N21[g,e,x,p];
			n21=Simplify[Outer[temp[[i,j,Sequence@@#2,Sequence@@#1]]&,combs2,combs1,1]];
			temp=N22[g,e,x,p];
			n22=Simplify[Outer[temp[[i,j,Sequence@@#2,Sequence@@#1]]&,combs2,combs2,1]];
			Transpose[Join[Transpose[Join[n11,n21]],Transpose[Join[Transpose[getZeros[dim,p+1]],n22]]]]
	];

(* Solving the kernel *)

solveKY[g_,e_,x_,p_Integer,v_]:=
	Block[{dim,eqky},
		dim=Length[x];
		eqky=Table[kcKY[g,e,x,i,j,p].v,{i,dim},{j,i-1}]//Flatten;
		Solve[eqky==0,v]
	];

solveCCKY[g_,e_,x_,p_Integer,v_]:=
	Block[{dim,eqccky},
		dim=Length[x];
		eqccky=Table[kcCCKY[g,e,x,i,j,p].v,{i,dim},{j,i-1}]//Flatten;
		Solve[eqccky==0,v]
	];

(* help command *)
helpKYupperbound:= Print["The functions are: kcKY[g,e,x,i,j,p], kcCCKY[g,e,x,i,j,p], solveKY[g,e,x,p,v] and solveCCKY[g,e,x,p,v]."]


End[]

EndPackage[]

helpKYupperbound

Print["Enter 'helpKYupperbound' for this list of functions"]

Print["Version Number: v1.2 (13rd August, 2014)"]

